# Rule Details: Gemini Memory Bank (Comprehensive Implementation Guide)

## Preface
This document is a *deep*, *detailed*, and *operational* implementation guide for the **Gemini Memory Bank** system. It instructs Gemini how to **read**, **create**, **update**, and **use** the Memory Bank files stored exactly in the `.kilocode/rules/memory-bank/` folder (note: trailing slash conventions are discussed below). The guide is strict: Gemini must **understand and adhere** to the rules and workflows described here at every task start and whenever memory-bank operations are involved.

> Important: This file is written to be fully compatible with the KiloCode Memory Bank instructions and the memory-bank-instructions.md supplied in the project repository. Any deviation from the exact `.kilocode/rules/memory-bank/` path is forbidden unless the user instructs otherwise.

---

## Table of Contents
1. Introduction & Core Philosophy
2. File-system conventions & path rules
3. Core Memory Bank files — contents & ownership
4. Initialization workflow — exhaustive checklist
5. Regular Task workflow — mandatory steps
6. Update workflow — when & how to run
7. Tasks documentation workflow (`tasks.md`)
8. Context management & token considerations
9. Error handling, edge cases & recovery
10. Consistency, verification & assertions
11. Audit logs & change summaries
12. Appendices:
    - A: `brief.md` default text (for empty brief.md)
    - B: Writing conventions for `.md` files (slash placement)
    - C: Example `context.md` and `architecture.md` templates
13. Final verification checklist
14. End-of-File marker

---

## 1. Introduction & Core Philosophy
- **Problem statement.** Gemini resets all internal session memory between conversations. This forces repeated context re-supply or expensive full repository analysis. Memory Bank provides a persistent, version-controlled, human-readable set of canonical documents that restore the project context for Gemini at session start.
- **Goal.** Ensure Gemini always begins work with accurate project context — loading `.kilocode/rules/memory-bank/` files — and never performs major project changes without first reading them.
- **Principles:** Accuracy, non-speculation, reproducibility, and conservatism:
  - **Accuracy:** Document real facts discovered in the repository (file paths, versions, configuration). Do not invent or assume.
  - **Non-speculation:** `context.md` must be factual and short—no creative or aspirational language.
  - **Reproducibility:** All changes to the codebase or Memory Bank must be verifiable and recorded.
  - **Conservatism:** Do not modify developer-maintained files (notably `brief.md`) without explicit permission. Suggest edits; do not write them directly.

---

## 2. File-system conventions & path rules
- **Canonical folder location:** `.kilocode/rules/memory-bank/`
  - Gemini **must** operate on this path only for reading, creating, or updating memory bank files.
  - Use the trailing slash when naming the folder in human-readable instructions where instructions previously used a trailing slash. Where memory-bank-instructions.md uses no trailing slash, Gemini may write files with or without trailing slash as filesystem allows — but the canonical reference must remain `.kilocode/rules/memory-bank/` in documentation and commands.
- **File formats:** All files must be Markdown (`.md`) encoded in UTF-8 without BOM.
- **File permissions and ownership:** When creating files, use standard project file permissions (owner write, group read) — follow the repository’s default umask. If unable to set permissions, log the issue in the audit log.
- **Path verification:** Before referencing a path in `architecture.md` or updating files, Gemini must verify that the path exists in the repository. If a referenced path does not exist, mark it as "Not found — verify" with the actual path listed.
- **No alternate folders.** Creating memory bank files in other folders (e.g., `.memory-bank/`, `/tmp/.kilocode/`, `docs/memory-bank/`) is prohibited unless the user explicitly instructs Gemini to migrate or to use a different path.

---

## 3. Core Memory Bank files — contents & ownership
Gemini must create and maintain (as appropriate) the following files inside the canonical folder. Ownership notes indicate who may edit the file by default.

### 3.1 brief.md — Developer-maintained (SOURCE OF TRUTH)
- **Ownership:** Developer. Gemini MUST NOT edit `brief.md` directly except when brief.md is absent or empty *and* the developer explicitly requested initialization behavior that creates it (see Section 4).
- **Purpose:** High-level project definition, scope, objectives, and significance.
- **If empty:** On initialization, Gemini **must** write the exact default text provided in Appendix A into `brief.md` (and nowhere else) when `brief.md` is empty or missing — but only as part of a formal `initialize memory bank` workflow. The default text is:
  > Provide a concise and comprehensive description of this project, highlighting its main objectives, key features, used technologies and significance. Then, write this description into a text file named appropriately to reflect the project's content, ensuring clarity and professionalism in the writing. Stay brief and short.
- **Structure & recommended fields:** Title, short description (2–4 sentences), technologies list, target users, constraints, and critical files/entry points. Keep brief.md concise (recommended ~100–300 words).

### 3.2 product.md — AI-maintained
- **Ownership:** Gemini (AI). Created and updated by Gemini based on `brief.md` and repository analysis.
- **Purpose:** Product vision, problem statement, target audience, key features, and success metrics.
- **Update frequency:** After significant product changes or during `initialize` / `update memory bank` workflows.
- **Content requirements:** Problem statement, solution summary, user flows, key features, success metrics, key UX constraints. Keep it actionable and tied to code findings when possible.

### 3.3 context.md — AI-maintained (MOST FREQUENTLY CHANGED)
- **Ownership:** Gemini (AI). This file is the live snapshot of current work.
- **Purpose:** Present state of development — what’s being worked on, recent changes, immediate next steps, blockers, and pending decisions.
- **Tone & constraints:** Strictly factual, brief, no speculation, no "would be nice" language.
- **Update rules:** Gemini must update `context.md` after *every* completed task or significant change. For minor editorial changes (typos, formatting), Gemini can update silently but should log the change in the audit log.
- **Required sections:** Current focus, recent changes (with statuses), next steps (prioritized), active issues & blockers, technical debt notes, pending decisions.

### 3.4 architecture.md — AI-maintained
- **Ownership:** Gemini (AI).
- **Purpose:** Concrete system architecture: modules, components, directory layout with **real file paths**, data flows, and decision rationale.
- **Absolute requirement:** Use **actual paths** from the codebase. Do not use placeholders like `src/main.js` unless that path exists.
- **Content requirements:** High-level architecture diagram (textual), component list with paths, integration points, critical implementation paths, and rationale for major technical decisions found in the repo.
- **When to update:** When code structure changes, new components are added, or integration points change. Also updated during `initialize` and `update memory bank` workflows.

### 3.5 tech.md — AI-maintained
- **Ownership:** Gemini (AI).
- **Purpose:** Technical environment: language versions, build tools, runtime versions, linters, test frameworks, CI/CD notes, and key dependencies with versions.
- **Content requirements:** Exact versions (from package.json, requirements.txt, Pipfile.lock, etc.), setup instructions, commands for local development, common pitfalls, and environment variables needed.
- **When to update:** If dependency versions change, new tools are introduced, or during `initialize` / `update memory bank` workflows.

### 3.6 Optional supportive files
- `tasks.md` — documented repetitive tasks and workflows.
- `progress.md` — feature completion, releases, and milestones.
- `integrations/`, `api/`, `testing/`, `deployment/` — subfolders for complex projects.
- **Ownership & updates:** Gemini writes and maintains these as needed, but must flag developer-owned content for review where appropriate.

---

## 4. Initialization workflow — exhaustive checklist
**Trigger phrase:** `"initialize memory bank"` (exact phrase recommended). The user may issue the command or perform it via UI. Initialization must only start after explicit user request.

### 4.0 Preconditions
- Confirm repository root location.
- Confirm `.kilocode/rules/memory-bank/` folder existence. If missing, create it as needed during initialization.
- Confirm `brief.md` presence. If absent or empty, follow rules in Section 3.1 to populate it with default text during initialization only.
- Use a high-capacity model for initialization (recommended). If model resources are limited, warn the user and proceed with limited scope, marking `Memory Bank: Partial` as status.
- Log start of initialization in audit log with timestamp and commit hash (if repo available).

### 4.1 Exhaustive analysis steps
Gemini must scan the repository thoroughly. This is the most critical step in the entire system. The analysis must include, at minimum:
1. **Source code scan:** Read all source files relevant to the primary language(s).
2. **Configuration scan:** package.json, pyproject.toml, requirements.txt, Gemfile, Dockerfiles, Makefile, CI configs.
3. **Build & scripts:** npm/yarn scripts, build pipeline definitions, Docker build contexts, and deployment scripts.
4. **Tests:** Unit, integration, and E2E tests. Record testing frameworks and test organization.
5. **Documentation:** README, CHANGELOG, design docs, and inline comments.
6. **Dependencies:** Identify direct and transitive dependencies and versions (use lockfiles where available).
7. **Runtime & infra:** Identify runtime environments (Node, Python, Java), database connections, external services, and environment variables.
8. **Data flows:** Map how data moves through the system: entry points, pipelines, persistence, API calls.
9. **Special cases:** Feature flags, experimental branches, or nonstandard build steps.
10. **Security considerations:** Authentication, secret handling, and permission boundaries if detectable.
11. **Tests & coverage:** Note presence/absence of CI checks and test coverage status.

For each of the above, produce structured notes and extract exact file paths and snippets when relevant. If a file is too large, record its path and a small representative excerpt and note where the excerpt came from.

### 4.2 Create & populate Memory Bank files
After analysis, create or update the core files in `.kilocode/rules/memory-bank/` with detailed findings.

- **product.md:** Synthesize the product vision from `brief.md` and repository artifacts.
- **context.md:** Populate with current focus (initial analysis), recent changes (found during scan), and prioritized next steps.
- **architecture.md:** Document directory structure (actual paths), component responsibilities, and critical flows. Include "Found in: `<path>`" markers for verification.
- **tech.md:** Record exact versions and setup commands. Provide quick-start commands to run the project locally.
- **tasks.md (optional):** Add common maintenance tasks discovered; e.g., "Add new API route", "Upgrade dependency X safely".

### 4.3 Post-initialization verification steps (user review)
- Report a clear initialization summary to the user with the status indicator at the top of your message:
  - `[Memory Bank: Active] Initialization complete.`
  - Include bullets: Project name, key technologies (with versions), architecture highlight, current focus, files created/updated.
- Ask the user to review `brief.md` and other files for correctness.
- Offer explicit diffs for each file created/updated.
- Request permission before making changes to developer-managed content (notably `brief.md`), except writing default text into empty `brief.md` as described above.
- Record the initialization completion time and save in the audit log and as a line in `progress.md` (if present).

---

# Continued in the file...

## 5. Regular Task workflow — mandatory steps
This workflow is executed at the **start of every task** (no exceptions). If any step fails, document the failure and provide options to continue with reduced guarantees.

### 5.1 Mandatory: Read ALL Memory Bank files
- Files to read (order matters):
  1. `brief.md` (foundation — read first)
  2. `context.md` (current state — read second)
  3. `architecture.md`
  4. `tech.md`
  5. `product.md`
  6. Optional: `tasks.md`, `progress.md`, subfolders
- If a file is missing or empty, record it in the status and behave according to Section 8 (error handling).

### 5.2 Synthesize & confirm understanding
After reading, Gemini must produce a one-line status at the top of its response, e.g.:
- `[Memory Bank: Active] I understand we are working on ...`
- `[Memory Bank: Partial] Found brief.md and context.md, but architecture.md missing.`
- `[Memory Bank: Missing] The folder .kilocode/rules/memory-bank/ does not exist.`
Then, provide a short (1–3 sentence) confirmation of understanding derived from `context.md` and `brief.md`.

### 5.3 Check tasks.md for matching workflows
If `tasks.md` lists a procedure matching the current task, Gemini must follow it and state:
> Matched task: `<task name>` from tasks.md — following documented steps.

### 5.4 Execute work using Memory Bank guidance
Use file paths and toolchains documented in `architecture.md` and `tech.md`.  
If new paths or files are created, record them in notes before execution.

### 5.5 Update `context.md` at task end
After completion, update `context.md` with:
- Completed steps  
- New blockers  
- New next steps  
- Decision logs

If repetitive, suggest adding it to `tasks.md`.

### 5.6 Status reporting
End with a summary containing `[Memory Bank: Active]` and mention which files were updated.

---

## 6. Update workflow — when & how to run
**Trigger:** `"update memory bank"` or major repository changes.

- Update when architecture, dependencies, or context changes.
- Perform a full scan (similar to initialization).  
- Do **not** modify `brief.md`; it is developer-owned.  
- Record diffs and timestamps in the audit log and progress.md.  
- Provide user summary of updated files and key findings.

---

## 7. Tasks documentation workflow (`tasks.md`)
**Trigger:** `add task` or similar instruction.

Each entry must include:  
- Task name  
- Description & prerequisites  
- Files modified (exact paths)  
- Step-by-step procedure  
- Verification notes and gotchas  

After saving, Gemini should confirm:
> Task saved to .kilocode/rules/memory-bank/tasks.md successfully.

---

## 8. Context management & token considerations
Memory Bank context is loaded each session to reduce redundant token use.  
If token limits approach, Gemini must summarize and store compressed state into `context.md` and request user approval.

---

## 9. Error handling, edge cases & recovery
- `[Memory Bank: Missing]` — Folder absent → Offer initialization.  
- `[Memory Bank: Partial]` — Missing files → List them and offer reconstruction.  
- Corrupted files → Backup old version (`.bak.TIMESTAMP.md`), rebuild from repo data.  
- Conflicts between `brief.md` and others → Always prioritize `brief.md` as canonical.  

---

## 10. Consistency, verification & assertions
Before writing any file, Gemini must verify:  
- File paths exist in repository.  
- Dependency versions match lockfiles.  
- Every claim traceable to source code.  
All unverified statements must be flagged with `"To verify"` notes.

---

## 11. Audit logs & change summaries
Maintain `.kilocode/rules/memory-bank/audit.log.md`.  
Each entry format:  
```
2025-10-16T10:22:11Z — initialize — user — created product.md, context.md
```
Major updates require corresponding `progress.md` or `changes.md` entry.

---

## 12. Appendices

### Appendix A — brief.md default text
```
Provide a concise and comprehensive description of this project, highlighting its main objectives, key features, used technologies and significance. Then, write this description into a text file named appropriately to reflect the project's content, ensuring clarity and professionalism in the writing. Stay brief and short.
```

### Appendix B — Folder naming convention
When memory-bank-instructions.md uses trailing slash, keep it.  
Canonical documentation reference must always be `.kilocode/rules/memory-bank/`.

### Appendix C — Example `context.md` template
```markdown
# Context — Current State

## Current Focus
- Bug fixing, module cleanup.

## Recent Changes
- Updated UI components (done).

## Next Steps
1. Improve test coverage.
2. Refactor auth middleware.

## Blockers
- Awaiting API schema update.
```

---

## 13. Final verification checklist
Before completing any operation, verify:  
- All five core files exist.  
- `brief.md` unmodified unless empty.  
- Paths validated and exist.  
- `audit.log.md` entry added.  
- Status `[Memory Bank: Active]` confirmed.

---

## 14. End-of-File marker
# End-of-File (EOF), 2025-10-16 10:13:56
